<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;

use App\Models\Dispositivo;
use App\Models\Areas;
use App\Models\TipoDispositivo;
use App\Models\Sucursal;
use App\Models\Estado;
use App\Models\Pantalla;
use App\Models\PcLapCel;

class InventarioController extends Controller
{
    // Muestra la vista principal del inventario con todos los dispositivos y datos relacionados
    public function mostrarinventario()
    {
        // Obtiene dispositivos con relaciones para mostrar datos completos
        $dispositivos = Dispositivo::with([
            'tipoDispositivo',
            'area',
            'sucursal',
            'estado',
            'pclapcel',
            'pantalla'
        ])->get();

        // Obtiene listas para los filtros y selects en la vista
        $tiposDispositivo = TipoDispositivo::all();
        $areas = Areas::all();
        $sucursales = Sucursal::all();
        $estados = Estado::all();

        // Retorna la vista con los datos compactados
        return view('inventario', compact('dispositivos', 'tiposDispositivo', 'areas', 'sucursales', 'estados'));
    }

    // Muestra la vista para agregar un nuevo dispositivo con datos para selects
    public function MostrarVistaAgregar(Request $request){
        $tiposDispositivo = TipoDispositivo::all();
        $estados = Estado::all();
        $areas = Areas::all();
        $sucursales = Sucursal::all();
        return view('agregar', compact('tiposDispositivo', 'estados', 'areas', 'sucursales'));
    }

    // Procesa la creación y almacenamiento de un nuevo dispositivo
    public function agregar(Request $request){
        // Validación de campos del formulario
        $request->validate([
            'NombreDispositivo' => 'required|string|max:30',
            'Marca' => 'required|string|max:30',
            'Modelo' => 'required|string|max:100',
            'IdEstado' => 'required|integer|exists:estado,IdEstado',
            'IdArea' => 'required|integer|exists:areas,IdArea',
            'IdSucursal' => 'required|integer|exists:sucursal,IdSucursal',

            // Campos opcionales para pc/laptop/celular
            'Memoria' => 'nullable|numeric|max:5000',
            'TipoMemoria' => 'nullable|string|max:30',
            'Ram' => 'nullable|numeric|max:100',
            'Procesador' => 'nullable|string|max:30',
            'SistemaOperativo' => 'nullable|string|max:30',

            // Campo opcional para pantalla
            'Pulgadas' => 'nullable|numeric|min:1'
        ]);

        // Crea el dispositivo principal con datos básicos
        $nuevoDispositivo = Dispositivo::create([
            'IdTipoDispositivo' => $request->Tipo,
            'NombreDispositivo' => $request->NombreDispositivo,
            'Marca' => $request-> Marca,
            'Modelo' => $request-> Modelo,
            'IdEstado' => $request-> IdEstado,
            'IdArea' => $request-> IdArea,
            'IdSucursal' => $request-> IdSucursal 
        ]);        
        
        // Si el tipo es pc, laptop o celular, crea registro en tabla pclapcel y vincula
        if(in_array($request->Tipo, [1,2,7])){
            $llenarTablaPcLapCel =  PcLapCel::create([
                'Memoria' => $request-> Memoria,
                'TipoMemoria' => $request-> TipoMemoria,
                'Ram' => $request-> Ram,
                'Procesador' => $request-> Procesador,
                'SistemaOperativo' => $request-> SistemaOperativo
            ]);

            $nuevoDispositivo->IdPcLapCel = $llenarTablaPcLapCel->IdPcLapCel;
            $nuevoDispositivo->save();
        }

        // Si el tipo es pantalla, crea registro en tabla pantalla y vincula
        if($request->Tipo == 3){
            $llenarTablaPantalla = Pantalla::create([
                'Pulgadas' => $request->Pulgadas
            ]);

            $nuevoDispositivo->IdPantalla = $llenarTablaPantalla->IdPantalla;
            $nuevoDispositivo->save();
        }

        // Redirecciona con mensaje de éxito o error
        if($nuevoDispositivo){
            return redirect()->route('mostrarinventario')->with('success', 'Dispositivo agregado correctamente');
        } else{
            return redirect()->route('mostrarinventario')->with('error', 'Error al registrar el dispositivo. Intenta nuevamente.');
        }
    }

    // Muestra la vista para actualizar un dispositivo con sus datos actuales
    public function MostrarVistaActualizar($id){
        $dispositivo = Dispositivo::with(['tipoDispositivo', 'area', 'sucursal', 'estado', 'pclapcel', 'pantalla'])->find($id);
        $areas = Areas::all();
        $sucursales = Sucursal::all();
        $estados = Estado::all();
        return view('actualizar', compact('dispositivo','areas', 'sucursales', 'estados'));
    }

    // Actualiza el dispositivo y sus relaciones con datos del formulario
    public function actualizar(Request $request, $id){
        // Validación igual que en agregar
        $request->validate([
            'NombreDispositivo' => 'required|string|max:30',
            'Marca' => 'required|string|max:30',
            'Modelo' => 'required|string|max:100',
            'IdEstado' => 'required|integer|exists:estado,IdEstado',
            'IdArea' => 'required|integer|exists:areas,IdArea',
            'IdSucursal' => 'required|integer|exists:sucursal,IdSucursal',

            'Memoria' => 'nullable|numeric|max:5000',
            'TipoMemoria' => 'nullable|string|max:30',
            'Ram' => 'nullable|numeric|max:100',
            'Procesador' => 'nullable|string|max:30',
            'SistemaOperativo' => 'nullable|string|max:30',
            'Pulgadas' => 'nullable|numeric|min:1'
        ]);

        // Busca el dispositivo a actualizar con sus relaciones
        $dispositivo = Dispositivo::with(['tipoDispositivo', 'area', 'sucursal', 'estado', 'pclapcel', 'pantalla'])->find($id);
        
        // Actualiza los datos básicos del dispositivo
        $dispositivo->update([
            'NombreDispositivo' => $request->NombreDispositivo,
            'Marca' => $request->Marca,
            'Modelo' => $request->Modelo,
            'IdEstado' => $request->IdEstado,
            'IdArea' => $request->IdArea,
            'IdSucursal' => $request->IdSucursal
        ]);

        // Actualiza datos de pc/laptop/celular si existen
        if($dispositivo->pclapcel){
            $dispositivo->pclapcel->update([
                'Memoria' => $request->Memoria,
                'TipoMemoria' => $request->TipoMemoria,
                'Ram' => $request->Ram,
                'Procesador' => $request->Procesador,
                'SistemaOperativo' => $request->SistemaOperativo
            ]);
        }

        // Actualiza datos de pantalla si existen
        if($dispositivo->pantalla){
            $dispositivo->pantalla->update([
                'Pulgadas' => $request->Pulgadas
            ]);
        }

        // Redirige con mensaje según resultado
        if($dispositivo){
            return redirect()->route('mostrarinventario')->with('success', 'Dispositivo actualizado correctamente');
        }else{
            return redirect()->route('mostrarinventario')->with('error', 'Error al actualizar el dispositivo. Intenta nuevamente.');
        }
    }

    // Elimina un dispositivo junto con sus relaciones (en cascada)
    public function EliminarRegistro($id)
    {
        // Busca dispositivo con relaciones
        $dispositivo = Dispositivo::with(['tipoDispositivo', 'area', 'sucursal', 'estado', 'pclapcel', 'pantalla'])->find($id);

        if($dispositivo){
            // Elimina el dispositivo (con eliminaciones en cascada definidas en modelo)
            $dispositivo->delete();
            return redirect()->route('mostrarinventario')->with('success', 'Dispositivo eliminado correctamente');
        } else{
            return redirect()->route('mostrarinventario')->with('error', 'Error al eliminar el dispositivo. Intenta nuevamente.');
        }
    }

    // Realiza búsqueda y filtrado en dispositivos según criterios enviados
    public function buscar(Request $request)
    {
        // Consulta dinámica con filtros condicionales
        $dispositivos = Dispositivo::with(['tipoDispositivo', 'area', 'sucursal', 'estado', 'pclapcel', 'pantalla'])
            ->when($request->filled('busqueda'), function ($query) use ($request) {
                $query->where('NombreDispositivo', 'like', "%{$request->busqueda}%");
            })
            ->when($request->filled('tipo'), function ($query) use ($request) {
                $query->where('IdTipoDispositivo', $request->tipo);
            })
            ->when($request->filled('area'), function ($query) use ($request) {
                $query->where('IdArea', $request->area);
            })
            ->when($request->filled('sucursales'), function ($query) use ($request) {
                $query->where('IdSucursal', $request->sucursales);
            })
            ->when($request->filled('estado'), function ($query) use ($request) {
                $query->where('IdEstado', $request->estado);
            })
            ->get();

        // Datos para selects de filtros
        $tiposDispositivo = TipoDispositivo::all();
        $areas = Areas::all();
        $sucursales = Sucursal::all();
        $estados = Estado::all();

        // Si no hay resultados, regresa con mensaje de error
        if($dispositivos->isEmpty()){
            return redirect()->route('mostrarinventario')->with('error', 'Ningún elemento coincide con su búsqueda.');
        }

        // Muestra la vista con resultados filtrados
        return view('inventario', compact('dispositivos', 'tiposDispositivo', 'areas', 'sucursales', 'estados'));
    }

}
