<?php
//  Este archivo define el modelo Dispositivo, que representa la tabla principal dispositivo dentro del sistema de inventario. 
//  Está configurado para trabajar con Laravel Eloquent ORM y facilita las operaciones CRUD sobre los registros de dispositivos. 
//  También se establece la relación con las tablas relacionadas y la eliminación en cascada de registros dependientes.
//  lo mismo aplica para los demás modelos.
namespace App\Models;

use Illuminate\Database\Eloquent\Model;

// Modelo que representa la tabla 'dispositivo'
class Dispositivo extends Model
{
    // Nombre de la tabla en la base de datos
    protected $table = 'dispositivo';

    // Clave primaria de la tabla
    protected $primaryKey = 'IdDispositivo';

    // Desactiva la gestión automática de timestamps (created_at, updated_at)
    public $timestamps = false;

    // Campos que se pueden asignar masivamente
    protected $fillable = [
        'IdTipoDispositivo',
        'NombreDispositivo',
        'Marca',
        'Modelo',
        'IdPcLapCel',
        'IdPantalla',
        'IdArea',
        'IdSucursal',
        'IdEstado',
    ];

    // Relación: Un dispositivo pertenece a un tipo de dispositivo
    public function tipoDispositivo()
    {
        return $this->belongsTo(TipoDispositivo::class, 'IdTipoDispositivo');
    }

    // Relación: Un dispositivo pertenece a un área
    public function area()
    {
        return $this->belongsTo(Areas::class, 'IdArea');
    }

    // Relación: Un dispositivo pertenece a una sucursal
    public function sucursal()
    {
        return $this->belongsTo(Sucursal::class, 'IdSucursal');
    }

    // Relación: Un dispositivo pertenece a un estado
    public function estado()
    {
        return $this->belongsTo(Estado::class, 'IdEstado');
    }

    // Relación: Un dispositivo pertenece a un registro de PC, Laptop o Celular
    public function pclapcel()
    {
        return $this->belongsTo(PcLapCel::class, 'IdPcLapCel');
    }

    // Relación: Un dispositivo pertenece a una pantalla
    public function pantalla()
    {
        return $this->belongsTo(Pantalla::class, 'IdPantalla');
    }

    // Evento para la eliminación en cascada de registros relacionados
    // Si se elimina un dispositivo, también se eliminan sus registros relacionados en pclapcel y pantalla
    public static function boot(){
        parent::boot();

        static::deleting(function ($dispositivo) {
            if ($dispositivo->pclapcel) {
                $dispositivo->pclapcel->delete();
            }
            if ($dispositivo->pantalla) {
                $dispositivo->pantalla->delete();
            }
        });
    }
}
